<?php

namespace App\Core;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

class TrendyolClient
{
    private $client;
    private $sellerId;
    private $baseUrl;

    public function __construct()
    {
        $this->sellerId = $_ENV['TRENDYOL_SELLER_ID'];
        $isStage = $_ENV['TRENDYOL_IS_STAGE'] === 'true';
        $this->baseUrl = $isStage ? "https://stageapigw.trendyol.com" : "https://apigw.trendyol.com";

        $apiKey = $_ENV['TRENDYOL_API_KEY'];
        $apiSecret = $_ENV['TRENDYOL_API_SECRET'];

        $this->client = new Client([
            'base_uri' => $this->baseUrl,
            'auth' => [$apiKey, $apiSecret],
            'verify' => false,
            'headers' => [
                'User-Agent' => $this->sellerId . " - SelfIntegration",
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ]
        ]);
    }

    public function request($method, $endpoint, $options = [])
    {
        try {
            $response = $this->client->request($method, $endpoint, $options);
            return json_decode($response->getBody()->getContents(), true);
        } catch (GuzzleException $e) {
            $response = $e->getResponse();
            $body = $response ? $response->getBody()->getContents() : 'No Body';
            throw new \Exception("Trendyol API Error: " . $e->getMessage() . " | Body: " . $body);
        }
    }

    // Ürün Servisleri
    public function getProducts($page = 0, $size = 50)
    {
        $endpoint = "/integration/product/sellers/{$this->sellerId}/products";
        return $this->request('GET', $endpoint, ['query' => ['page' => $page, 'size' => $size]]);
    }

    public function createProducts($items)
    {
        $endpoint = "/integration/product/sellers/{$this->sellerId}/products";
        return $this->request('POST', $endpoint, ['json' => ['items' => $items]]);
    }

    public function updateProducts($items)
    {
        $endpoint = "/integration/product/sellers/{$this->sellerId}/products";
        return $this->request('PUT', $endpoint, ['json' => ['items' => $items]]);
    }

    public function deleteProducts($items)
    {
        $endpoint = "/integration/product/sellers/{$this->sellerId}/products";
        return $this->request('DELETE', $endpoint, ['json' => ['items' => $items]]);
    }

    public function updatePriceAndInventory($items)
    {
        $endpoint = "/integration/inventory/sellers/{$this->sellerId}/products/price-and-inventory";
        return $this->request('POST', $endpoint, ['json' => ['items' => $items]]);
    }

    // Sipariş Servisleri
    public function getOrders($params = [])
    {
        $endpoint = "/integration/order/sellers/{$this->sellerId}/orders";
        $defaultParams = [
            'page' => 0,
            'size' => 50,
            'orderByField' => 'PackageLastModifiedDate',
            'orderByDirection' => 'DESC'
        ];
        $queryParams = array_merge($defaultParams, $params);
        return $this->request('GET', $endpoint, ['query' => $queryParams]);
    }

    public function getBatchStatus($batchId)
    {
        $endpoint = "/integration/product/sellers/{$this->sellerId}/products/batch-requests/{$batchId}";
        return $this->request('GET', $endpoint);
    }

    public function getBrands($name = null, $page = 0, $size = 50)
    {
        if ($name) {
            $endpoint = "/integration/product/brands/by-name";
            return $this->request('GET', $endpoint, ['query' => ['name' => $name]]);
        }
        $endpoint = "/integration/product/brands";
        return $this->request('GET', $endpoint, ['query' => ['page' => $page, 'size' => $size]]);
    }

    public function getCategories()
    {
        $endpoint = "/integration/product/product-categories";
        return $this->request('GET', $endpoint);
    }
}
